clear all
clear matrix
capture log close
set max_memory 20g
set min_memory 5g
set matsize 1000
set maxvar 30000
set more off

/* ----------------------------------------------------------------------------- *
Replication file for "Measuring Financial Inclusion: Explaining Variation Across 
and Within Countries" by Asli Demirguc-Kunt and Leora Klapper (2013)

Structure of do-file:
	- A: User Actions
	- B: Merge in public country-level data
	- C: Create/transform variables
	- D: Replication code
 * ----------------------------------------------------------------------------- */


/* ----------------------------------------------------------------------------- *
IMPORTANT NOTE: The Global Findex data come from a survey carried out by Gallup,
Inc. as part of the annual Gallup World Poll. A prior confidentiality agreement
with Gallup, Inc. prevents us from making publicly available several series used
in this paper. Without these series, users will not be able to produce the
complete set of empirical results in the paper. 

     Series that are not publicly available:
	- psu
	- strata
	- rural  	(rural vs urban residency)
	- hhs		(log of household size)
	- empl_1	(employment category: employed for employer)
	- empl_2	(employment category: unemployed) 
	- empl_3	(employment category: out of workforce)
	- empl_4	(excluded employment category)
	- inc_c 	(continuous income variable)
	- married 	(marital status)
	- divorced  (maritual status)
	
	Data for Iran, Islamic Rep. is not publicly available. 
	
* ----------------------------------------------------------------------------- */

/* ----------------------------------------------------------------------------- *
    A: User Actions: Select folders and figures/maps/regressions to replicate
* ----------------------------------------------------------------------------- */

*STEP ONE: Edit working directory (line 52) and create new subfolder for these results (line 54)

local cd1 "\\tsclient\Z\Brookings"
cd `cd1'  /* change working directory */
local newdir "Replication3"
mkdir "`newdir'"

/*STEP TWO: Select Figures, Maps, and Regressions to replicate. Choices are in parentheses. 
Those marked with a * cannot be replicated as they use non-public data. */

*Figures (1,2,3,4*,5,6*,7,8,9,10,11,12,13,14): Place desired figures after "flist" in line below. 
local flist 1,2,3,5,7,8,9,10,11,12,13,14
*Regressions (1,2*): Place desired regressions after "rlist" in line below. 
local rlist 1


*STEP THREE: Run do-file. Results will output into subfolder.

use "micro_WORLD.dta", clear

/* ----------------------------------------------------------------------------- *
    B: Merge in public country-level data
* ----------------------------------------------------------------------------- */

merge m:1 economy using "economy level_public"
drop _merge

/* ----------------------------------------------------------------------------- *
    C: Create/transform variables
* ----------------------------------------------------------------------------- */

*Transform Relevant Findex variables into binary yes/no
foreach var of varlist account q10a-q10g q13a q14a-q14e q15a-q15e { 
recode `var' (1 = 1 "yes")(2/4 = 0 "no/dk/ref") ,pre(yn_) l(yesno)
}

replace yn_q13a = 0 if yn_q13a==. & account!=. // gives "0" to all those who don't save
gen mobileb = (q15a1a==1 | q15a1b==1 | q15a1c==1) // any use of mobile phone to send money
replace mobileb=. if q15a1a==.

gen yn_q10eonly = (yn_q10e==1 & yn_q10a==0 & yn_q10b==0 & yn_q10c==0 & yn_q10d==0 & yn_q10f==0 & yn_q10g==0)
replace yn_q10eonly=. if yn_q10e==.

*Transform individual characteristic variables
gen		age2 = age * age
label var age2	"Age squared"
	
gen		inc_poor = 0 if inc_q != .
replace inc_poor = 1 if inc_q == 1
gen		inc_2nd  = 0 if inc_q != .
replace inc_2nd  = 1 if inc_q == 2
gen 	inc_3rd  = 0 if inc_q != .
replace inc_3rd  = 1 if inc_q == 3
gen 	inc_4th	 = 0 if inc_q != .
replace inc_4th	 = 1 if inc_q == 4
gen		inc_rich = 0 if inc_q != .
replace inc_rich = 1 if inc_q == 5
label var inc_poor "income quintile: poorest 20%"
label var inc_2nd  "income quintile: second 20%"
label var inc_3rd  "income quintile: middle 20%"
label var inc_4th  "income quintile: fourth 20%"
label var inc_rich "income quintile: richest 20%"


gen		edu_1st = educ
replace edu_1st = . if edu_1st == 4 | edu_1st == 5
replace edu_1st = 0 if edu_1st != 1 & edu_1st != .
gen		edu_2nd = educ
replace edu_2nd = . if edu_2nd == 4 | edu_2nd == 5
replace edu_2nd = 0 if edu_2nd != 2 & edu_2nd != .
replace edu_2nd = 1 if edu_2nd == 2 
gen		edu_3rd = educ
replace edu_3rd = . if edu_3rd == 4 | edu_3rd == 5
replace edu_3rd = 0 if edu_3rd != 3 & edu_3rd != .
replace edu_3rd = 1 if edu_3rd == 3 
gen 	edu = 1 if edu_1st == 1
replace edu = 2 if edu_2nd == 1
replace edu = 3 if edu_3rd == 1
label var edu_1st "Education: completed elementary education or less (up to 8 years of basic education)"
label var edu_2nd "Education: secondary - 3 year tertiarysecondary education and some education beyond secondary education (9-15 years of education)"
label var edu_3rd "Education: completed four years of education beyond high school and/or received a 4-year college degree."


*Create global weights (and correct for variation in # obs per country)
gen pop_adult_worldshare = pop_adult/5020736222.1 // each country's share of global adult population
bysort economy: egen obs = count(wgt) // observations per country
gen rep_pop_adult_worldshare = pop_adult_worldshare/obs //  share of global adult population represented by each observation

gen wgt_global = wgt*rep_pop_adult_worldshare // global weight


*Exclude countries with sampling exclusions >20%

/*Note: The worldwide aggregates omit countries for which Gallup excludes more than 20 percent 
of the population in the sampling either because of security risks or because the population 
includes non-Arab expatriates. These excluded countries are Algeria, Bahrain, the Central 
African Republic, Madagascar, Qatar, Somalia, and the United Arab Emirates. The Islamic 
Republic of Iran is also excluded because the data were collected in that country using a 
methodology inconsistent with that used for other countries (the survey was carried out by 
phone from Turkey). The exclusion of the Islamic Republic of Iran has a nontrivial effect on 
regional aggregates because its population is larger and wealthier than those of other countries
in the Middle East and North Africa. For example, account penetration in the region is
estimated to be 18 percent when the Islamic Republic of Iran is excluded but 33 percent when it
is included. These countries have missing values for variables regionwb2 and incomegroupwb2*/

gen regionwb2 = regionwb
replace regionwb2 = ""  if economy=="Algeria" |  economy=="Bahrain" | economy=="Central African Republic" | economy=="Iran, Islamic Rep." | economy=="Madagascar" | economy=="Qatar" | economy=="Somalia" | economy=="United Arab Emirates"

gen incomegroupwb2=incomegroupwb
replace incomegroupwb2 = ""  if economy=="Algeria" |  economy=="Bahrain" | economy=="Central African Republic" | economy=="Iran, Islamic Rep." | economy=="Madagascar" | economy=="Qatar" | economy=="Somalia" | economy=="United Arab Emirates"
	// creating income group dummies
gen		d_low = 0 		if incomegroupwb2 != ""
replace d_low = 1 		if incomegroupwb2 == "Low income"
gen		d_lowerm = 0 	if incomegroupwb2 != ""
replace d_lowerm = 1 	if incomegroupwb2 == "Lower middle income"
gen		d_upperm = 0 	if incomegroupwb2 != ""
replace d_upperm = 1 	if incomegroupwb2 == "Upper middle income"
gen		d_high = 0 		if incomegroupwb2 != ""
replace d_high = 1 		if incomegroupwb2 == "High income"
gen		d_devel = 0 	if incomegroupwb2 != ""
replace d_devel = 1 	if incomegroupwb2 != "" & incomegroupwb2 != "High income"

/* ----------------------------------------------------------------------------- *
    D: Replication code
* ----------------------------------------------------------------------------- */

*Figure 1: Has an Account at a Formal Financial Institution, by Income Group
if inlist(1,`flist') {

preserve
keep if incomegroupwb2!=""
collapse yn_account [pweight=wgt_global], by(incomegroupwb2)
outsheet using "`newdir'\Figure1.xls"
restore

}

*Figure 2 / Map 1: Has an Account at a Formal Financial Institution, by Economy
if inlist(2,`flist') {

preserve
collapse yn_account [pweight=wgt], by(economy)
outsheet using "`newdir'\Figure2.xls"
restore

}


*Figure 3: Financial Inclusion, GDP Per Capita, and Financial Development, All countries
if inlist(3,`flist') {

preserve
collapse yn_account dctps gdppc_usd_c [pweight=wgt], by(economy)
outsheet using "`newdir'\Figure3.xls"
restore

}

*Figure 4: Has an Account at a Formal Financial Institution, by country-level income and within-country income quintiles
if inlist(4,`flist') {

preserve
keep if incomegroupwb2!=""
sort incomegroupwb2 inc_q
egen id = group(incomegroupwb2 inc_q)
collapse (first) incomegroupwb2 inc_q (median) inc_c (mean) yn_account [pweight=wgt_global], by(id)
drop if inc_q==.
gen ln_inc_c = ln(inc_c)
outsheet using "`newdir'\Figure4.xls"
restore


}

*Figure 5: Adults in the poorest income quintile without an account at a formal financial institution (%)
if inlist(5,`flist') {

preserve
keep if economy=="Australia" | economy=="Canada" | economy=="United Kingdom" | economy=="United States"   
keep if inc_q==1 // keep only those in lowest country-level income quintile
collapse yn_account [pweight=wgt], by(economy)
gen noaccount = 1-yn_account
drop yn_account
outsheet using "`newdir'\Figure5.xls"
restore

}

*Figure 6: Saves at a Formal Financial Institution, by country-level income and within-country income quintiles
if inlist(6,`flist') {

preserve
keep if incomegroupwb2!=""
sort incomegroupwb2 inc_q
egen id = group(incomegroupwb2 inc_q)
collapse (first) incomegroupwb2 inc_q (median) inc_c (mean) yn_q13a [pweight=wgt_global], by(id)
drop if inc_q==.
gen ln_inc_c = ln(inc_c)
outsheet using "`newdir'\Figure6.xls"
restore


}

*Figure 7 / Map 2: Origination of new formal loans around the world
if inlist(7,`flist') {

preserve
collapse yn_q14a [pweight=wgt], by(economy)
outsheet using "`newdir'\Figure7.xls"
restore

}

*Figure 8: Sources of new formal and informal loans, (%)
if inlist(8,`flist') {

preserve
keep if incomegroupwb2!=""
collapse yn_q14a yn_q14b yn_q14c yn_q14e [pweight=wgt_global], by(incomegroupwb2)
outsheet using "`newdir'\Figure8.xls"
restore

}


*Figure 9 / Map 3: Currently has loan for home purchase
if inlist(9,`flist') {

preserve
collapse yn_q15a [pweight=wgt], by(economy)
outsheet using "`newdir'\Figure9.xls"
restore

}



*Figure 10: Purpose of outstanding loans reported by borrowers in developing countries (%) 
if inlist(10,`flist') {

preserve
keep if incomegroupwb2!="High income" & incomegroupwb2!=""
collapse yn_q15b yn_q15c yn_q15d yn_q15e [pweight=wgt_global]
outsheet using "`newdir'\Figure10.xls"
restore

}

*Figure 11: Reported Reasons for not having a Bank Account
if inlist(11,`flist') {

preserve
collapse yn_q10a yn_q10b yn_q10c yn_q10d yn_q10e yn_q10eonly yn_q10f yn_q10g [pweight=wgt_global]
outsheet using "`newdir'\Figure11.xls"
restore

}

*Figure 12: Subjective vs. Objective Measures of Cost as a Barrier to Access
if inlist(12,`flist') {

preserve
collapse yn_q10b cost_npc2 [pweight=wgt], by(economy)
replace cost_npc2=2 if cost_npc2>2 & cost_npc2<2.5
replace cost_npc2=3 if cost_npc2>=2.5 & cost_npc2<3
replace cost_npc2=4 if cost_npc2>=3.5 & cost_npc2<4
collapse yn_q10b, by(cost_npc2)
drop if cost_npc2==.
outsheet using "`newdir'\Figure12.xls"
restore

}

*Figure 13: Subjective vs. Objective Measures of Documentation and cost as a Barrier to Access
if inlist(13,`flist') {

preserve
collapse pfa_doc_c	yn_q10c [pweight=wgt], by(economy)
drop if pfa_doc_c==.
outsheet using "`newdir'\Figure13.xls"
restore

}

*Figure 14 / Map 3: Use of Mobile Money in Sub-Saharan Africa
if inlist(14,`flist') {

preserve
keep if regionwb2=="Sub-Saharan Africa" | economy=="Egypt, Arab Rep." | economy=="Djibouti" | economy=="Morocco"
collapse mobileb [pweight=wgt], by(economy)
outsheet using "`newdir'\Figure14.xls"
restore

}

*Regression 1: Economy-level Regressions
if inlist(1,`rlist') {

gen 	y = ln(gdppc_usd_c) // using lagged value here
rename	dctps credit // using lagged value here 
replace credit = credit / 100
replace gini = gini / 100

collapse (mean) d_low d_lowerm d_upperm d_high d_devel yn_account yn_q13a y yn_q14a credit gini [pweight=wgt_global] if regionwb2!="", by(economy)

label var d_low "Income group: Low (World Bank)"
label var d_lowerm "Income group: Lower Middle (World Bank)"
label var d_upperm "Income group: Upper Midde (World Bank)"
label var d_high "Income group: High (World Bank)"
lab var y "log of GDP per capita (2010, WDI)"
lab var credit "Domestic credit to the private sector (% of GDP, 2010, WDI)"
lab var gini  "Gini index, latest available (WDI)"

foreach x in yn_account {

regr `x' y
outreg2 using `newdir'\Reg2.xls, label pfmt(fc) bdec(3) sdec(3)
regr `x' d_low d_lowerm d_upperm
test d_low == d_lowerm
local p1 = r(p)
test d_lowerm == d_upperm
local p2 = r(p) 
outreg2 using `newdir'\Reg2.xls, addstat(low_eq_lowerm p-value, `p1', lowerm_eq_upperm, `p2') label pfmt(fc) bdec(3) sdec(3) adec(3)
}
foreach x in yn_account {
regr `x' y credit
outreg2 using `newdir'\Reg2.xls, label pfmt(fc) bdec(3) sdec(3)
regr `x' y gini
outreg2 using `newdir'\Reg2.xls, label pfmt(fc) bdec(3) sdec(3)
}
foreach x in yn_q13a yn_q14a {
quietly regr `x' d_low d_lowerm d_upperm
test d_low == d_lowerm
local F1 = r(F)
local p1 = r(p)
test d_lowerm == d_upperm
local F2 = r(F)
local p2 = r(p) 
outreg2 using `newdir'\Reg2.xls, addstat(F statistic 1, `F1', p-value 1, `p1', F statistic 2, `F2', p-value 2, `p2') label pfmt(fc) bdec(3) sdec(3) adec(3)
}


}

*Regression 2: Individual-level Regressions
if inlist(2,`rlist') {

replace psu = 1 if psu == .	// so the singleunit(centered) option works, in the cases where there is just one PSU per strata, Gallup coded the PSU as missing, but they should have a value
svyset, clear				// clear survey setting
svyset  economy [pweight=wgt] || psu, strata(strata) singleunit(centered) // note: clustered at the country level

local rhs "i.economy female inc_poor inc_2nd inc_3rd inc_4th age age2 rural edu_1st hhs married divorced empl_1 empl_2 empl_3"

mat t=J(3,12,.)
local j = 1
local k = 1

foreach y in yn_account yn_q13a yn_q14a {


	local i=1
	foreach x in low lowerm upperm high {


	xi: svy: probit   `y' `rhs'	 if d_`x'==1
	estimates store a`y'`i'coef
	
	test inc_poor=inc_2nd
	local p1 = r(p)
	
	test inc_2nd=inc_3rd
	local p2 = r(p)
	
	test inc_3rd=inc_4th
	local p3 = r(p)	
		
	outreg2 using `newdir'\Reg1.xls, drop(o.* _Ie*) addstat(incpoor=inc_2nd, `p1', inc_2nd=inc_3rd, `p2', inc_3rd=inc_4th, `p3') label pfmt(fc) bdec(3) sdec(3) adec(3)
	
	margins, dydx(inc_poor inc_2nd inc_3rd inc_4th) post
	outreg2 using `newdir'\Reg1.xls, drop(o.* _Ie*) label dec(3)

	}
}	

}










